#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#include <fcntl.h>
#include <ctype.h>
#include <asm/types.h>
#include <sys/ioctl.h>
#include <sys/wait.h>
#include <sys/mman.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/ether.h>
#include <net/if.h>
#include <arpa/inet.h>
#include <net/route.h>
#include <asm/types.h>
#include <linux/netlink.h>
#include <linux/rtnetlink.h>
#include "fwparam_ibft.h"
#include "iscsi-ioctl.h"

#ifndef TRUE
#define TRUE 0
#endif

#ifndef FALSE
#define FALSE 1
#endif

#ifndef BOOL
#define BOOL int
#endif

#define ISCSIDEV "/dev/iscsictl"
static int iscsifd = -1;

extern struct ibft_nic *nic0;
extern struct ibft_nic *nic1;
extern struct ibft_tgt *tgt0;
extern struct ibft_tgt *tgt1;
extern struct ibft_control *control;
extern struct ibft_initiator *initiator;
extern char *ibft_loc;
int calculate_ibft_loc (void);

typedef struct optionRomHeader_s 
{
  unsigned short uiSignature; /* 55AA */
  unsigned char  ucSize;
  unsigned char  ucInit[4];
  unsigned char  ucCksum;
  unsigned char  iScsiSignature[6]; /* $iSCSI */
  unsigned short uiConfigData; 
  unsigned int	 uiUndiOffSet;
  unsigned short Dataebdaoffset;
  unsigned char  UndiCodesize;
  unsigned char  UndiDatasize;
  unsigned short uiPci;
  unsigned short uiPnp;
  unsigned short uiRelocTblOff;
  unsigned short uiRelocTblSeg;
  unsigned short UndiDataStart;
} optionRomHeader_t;

typedef struct
{
  char ifname[IFNAMSIZ]; 

  BOOL enabled;
  BOOL primary;

  unsigned long n_ip;
  unsigned long n_mac;  
  unsigned long n_net;
  unsigned long n_gate;

} netdev_t;

typedef struct
{
  unsigned long table;
  unsigned long i_iqn;

  netdev_t nic_0;
  netdev_t nic_1;

  unsigned long t_ip;
  unsigned long t_port;
  unsigned long t_iqn;

  BOOL chap;
  unsigned long  user;
  unsigned long  secret;

} offset_t;

#define OFFSET_TO_STRING(off) ((char*)off)

#define OPTION_ROM_SIGNATURE         0xAA55
#define START_FROM                   0xC0000
#define SEARCH_TILL                  0xF0000
#define MAP_SIZE                     (SEARCH_TILL - START_FROM + 1)

#define OPTION_ROM_HEADER_ALIGNMENT  512

// Speced length is 223 + 1 for the NULL char
#define MAX_IQN_LEN                  224
#define NUM_ETHERNET_INTERFACES       32
#define MAX_NETDEV                    20
#define MAC_LENGTH                     6
#define MAX_NETMASK_LENGTH            16

#define ETHSETTINGS_FILE             "/etc/ethsettings"

#define ETH_MASK                     0x01
#define DISCOVERY_MASK               0x02
#define IF_MASK                      0x04
#define SILENT_MASK                  0x08

#define PRINT_DBG_MSG do {							\
    fprintf(stderr, "iscsi_util:iscsi boot failure at line %u, file %s (%u) [%s]\n", \
	    __LINE__, __FILE__,						\
	    errno, strerror(errno)); exit(1); } while(0)

#define iscsi_printf(fmt, args...) if (SILENT_MASK != (flags & SILENT_MASK)) { printf("iSCSI Boot Utility: "fmt, ##args); }

void printValues(const offset_t* offset);
BOOL ipToAddress(const u_int32_t* ip, char* address);
BOOL prefixToNetmask(u_int32_t prefix, char* netmask);
BOOL writeNetworkInformation(const netdev_t* nic);
BOOL determineTableOffset(const unsigned long  map_base, offset_t* offset);
BOOL ifup(const netdev_t* nic);
BOOL determineInitiatorOffsets(offset_t* offset);
BOOL determineTargetOffsets(offset_t* offset);
int iscsiIoctl ( int request, void *data ); 
void iscsiSetDefaults ( struct iscsi_session_ioctl *session ); 
int connectToTarget ( const offset_t* offset ); 

u_int8_t flags       = 0;
static int sockfd = -1;

int main(int argc, char **argv) 
{
  int      fd          = 0;
  unsigned long      map_base    = 0;
  offset_t offset;
  int      option      = 0;

  while ((option = getopt (argc, argv, "ecduas")) != -1)
    {
      switch (option)
	{  
	case 'e':
	  flags |= ETH_MASK;
	  break;

	case 'u':
	  flags |= IF_MASK;
	  break; 

	case 's':
	  flags |= SILENT_MASK;
	  break;

	case 'd':
	  flags |= DISCOVERY_MASK;
	  break;

	case 'a':
	default:
	  flags |= ETH_MASK;
	  flags |= IF_MASK;
	  flags |= DISCOVERY_MASK;
	}
    }

  memset(&offset, 0, sizeof(offset_t));
  
  calculate_ibft_loc ();
 
  if (FALSE == determineTableOffset(map_base, &offset))
    {
      iscsi_printf("Unable to determine the table offset\n");
      close(fd);
      return FALSE;
    }
  
  if (FALSE == determineInitiatorOffsets(&offset))
    {
      iscsi_printf("Initiator was not configured correctly\n");
    }
  
  if (FALSE == determineTargetOffsets(&offset))
    {
      iscsi_printf("Target was not configured correctly\n");
      close(fd);
    }
 
   printValues(&offset);

  // It's horrible, but this application will continue to write duplicate entries
  // into the ethsettings file. I'm just going to > /etc/ethsettings before running
  // this application, but it really should be fixed here.
  if (ETH_MASK == (flags & ETH_MASK))
    {
      if (TRUE == offset.nic_0.enabled)
	{
	  if (TRUE == offset.nic_0.primary)
	    {
	      writeNetworkInformation(&(offset.nic_0));
	      
	      if (TRUE == offset.nic_1.enabled)
		{
		  writeNetworkInformation(&(offset.nic_1));
		}
	    }
	}
      
      if (TRUE == offset.nic_1.enabled)
	{
	  if (TRUE == offset.nic_1.primary)
	    {
	      writeNetworkInformation(&(offset.nic_1));
	      
	      if (TRUE == offset.nic_0.enabled)
		{
		  writeNetworkInformation(&(offset.nic_0));
		}
	    }
	}
    }

  if (IF_MASK == (flags & IF_MASK))
    {
      if (TRUE == offset.nic_0.enabled)
	{
	  ifup(&(offset.nic_0));
	}
      
      if (TRUE == offset.nic_1.enabled)
	{
	  ifup(&(offset.nic_1));
	}
    }

  // Do discovery
  if ( flags & DISCOVERY_MASK )
  {
     iscsi_printf("\nStarting Target Discovery...");
     connectToTarget(&offset);
  }

  // cleanup 
  munmap((void*)map_base, MAP_SIZE); 
  close(fd);
  return TRUE;
}

BOOL writeNetworkInformation(const netdev_t* nic)
{
  FILE* fd_ethsettings;
  int index;
  int sock;
  char iscsi_mac[200];
  char iface_mac[200];
  char ethX[IFNAMSIZ];
  struct ether_addr iscsi_eth;
  struct ether_addr eth;
  struct ifreq ifr;

  memset(&iscsi_eth, 0, sizeof(struct ether_addr));
  memcpy(iscsi_eth.ether_addr_octet, (char*)(nic->n_mac), MAC_LENGTH);

  sock = socket(PF_INET,SOCK_DGRAM, 0);
  
  if (-1 == sock)
    {
      iscsi_printf("Could not create a socket\n");
    }

  for(index = 0 ; index < NUM_ETHERNET_INTERFACES ; index++) 
    {
      memset(ethX, 0, IFNAMSIZ);
      memset(&ifr, 0, sizeof(struct ifreq));
      sprintf(ethX, "eth%d", index);
      strcpy(ifr.ifr_name, ethX);      
      
      if(0 > ioctl(sock, SIOCGIFHWADDR, &ifr))
	{
	  iscsi_printf("Could not get network device properties\n");
	}
      
      memset(&eth, 0, sizeof(struct ether_addr));
      memcpy(eth.ether_addr_octet, (char*)(ifr.ifr_hwaddr.sa_data), MAC_LENGTH);
      
      strcpy(iscsi_mac, ether_ntoa(&iscsi_eth));
      strcpy(iface_mac, ether_ntoa(&eth));
      
      if (0 == strcmp(iscsi_mac, iface_mac))
	{
	  strcpy((char*)nic->ifname, ifr.ifr_name);

	  if ((fd_ethsettings = fopen(ETHSETTINGS_FILE, "a")) != NULL)
	    {
	      fprintf(fd_ethsettings, "%s\n", ethX);
	      fflush(fd_ethsettings);
	      fclose(fd_ethsettings);
	    }
	  else
	    {
	      iscsi_printf("Failed to open %s\n", ETHSETTINGS_FILE);
	    }
	  break;
	}
    }
  
  return TRUE;
}

int connectToTarget (const offset_t* offset) 
{
    struct iscsi_session_ioctl session;
    char   address[MAX_NETMASK_LENGTH];
    char  *target_ip; 
    struct sockaddr_in *sin =
           ( struct sockaddr_in * ) &session.portal.addr;

    if(!offset)
    {
       printf("\noffset NULL, session connect abort");
       return -1;
    }
    memset ( &session, 0, sizeof ( session ) );

    iscsiSetDefaults ( &session );
    session.ioctl_version = ISCSI_SESSION_IOCTL_VERSION;

    if (TRUE == offset->chap)
    {
       session.password_length = strlen (OFFSET_TO_STRING(offset -> secret));

       strncpy ( ( char * ) session.username, OFFSET_TO_STRING(offset -> user),
              sizeof ( session.username ) );
       strncpy ( ( char * ) session.password, OFFSET_TO_STRING(offset -> secret),
              sizeof ( session.password ) );
    }

    strncpy ( ( char * ) session.target_name, OFFSET_TO_STRING(offset->t_iqn),
              sizeof ( session.target_name ) );
    strncpy ( ( char * ) session.initiator_name, OFFSET_TO_STRING(offset->i_iqn),
              sizeof ( session.initiator_name ) );
    strncpy ( ( char * ) session.initiator_alias, OFFSET_TO_STRING(offset->i_iqn),
              sizeof ( session.initiator_alias ) );

    sin->sin_family = AF_INET;

    sprintf(address, "%u.%u.%u.%u\n", ((u_int8_t*)(offset->t_ip))[12],
                                      ((u_int8_t*)(offset->t_ip))[13],
                                      ((u_int8_t*)(offset->t_ip))[14], 
                                      ((u_int8_t*)(offset->t_ip))[15]);
    target_ip =address;
    inet_aton(target_ip, &(sin->sin_addr));

    sin->sin_port = htons( *((u_int16_t*)(offset->t_port)));

    return iscsiIoctl ( ISCSI_ESTABLISH_SESSION, &session );
}

void iscsiSetDefaults ( struct iscsi_session_ioctl *session ) 
{
    static const char isid[6] = { 0x40, 0x00, 0x27, 0x23, 0x00, 0x00 };

    memset ( session, 0, sizeof ( *session ) );

    session->ioctl_version = ISCSI_SESSION_IOCTL_VERSION;
    memcpy ( session->isid, isid, sizeof ( session->isid ) );

    session->portal.tag = PORTAL_GROUP_TAG_UNKNOWN;
}

int iscsiIoctl ( int request, void *data ) 
{
    if ( iscsifd < 0 ) 
    {
        iscsifd = open ( ISCSIDEV, O_RDWR );
        if ( iscsifd < 0 )
           return -1;
    }

    return ( ioctl ( iscsifd, request, data ) );
}

BOOL determineTableOffset(const unsigned long  map_base, offset_t* offset)
{  
  offset->table = *((unsigned long*)ibft_loc);
  return TRUE;
}

BOOL ifup(const netdev_t* nic)
{
  struct ifreq       ifr;
  struct sockaddr_in sin;
  int                sock = 0;
  char               netmask[MAX_NETMASK_LENGTH];
  char               address[MAX_NETMASK_LENGTH];
  char               g_address[MAX_NETMASK_LENGTH];
  struct rtentry     route;
  struct sockaddr_in sindst,singw;
  
  memset(netmask, 0, MAX_NETMASK_LENGTH);

  if (0 >= strlen(nic->ifname))
    {
      iscsi_printf("The interface name is empty\n");
    }

  sock = socket(PF_INET,SOCK_DGRAM, 0);
  
  if (-1 == sock)
    {
      iscsi_printf("Could not create a socket\n");
      return FALSE;
    }

  memset(&ifr, 0, sizeof(struct ifreq));
  memset(&sin, 0, sizeof(struct sockaddr_in));
  strcpy(ifr.ifr_name, nic->ifname);

  sin.sin_family = AF_INET;
  sin.sin_port = 0;

  ipToAddress(&(((u_int32_t *)nic->n_ip)[3]), address);

  inet_aton(address, &(sin.sin_addr));

  memcpy(&ifr.ifr_addr, &sin, sizeof(struct sockaddr));

  iscsi_printf("Starting interface %s with address: %s\n", ifr.ifr_name, inet_ntoa(sin.sin_addr));

  // Set the IP address
  if (0 > ioctl(sock, SIOCSIFADDR, &ifr))
    {
      iscsi_printf("Could not set the IP address!\n");
      close(sock);
      return FALSE;
    }

  // Configure the netmask
  memset(&ifr, 0, sizeof(struct ifreq));
  memset(&sin, 0, sizeof(struct sockaddr_in));
  strcpy(ifr.ifr_name, nic->ifname);

  sin.sin_family = AF_INET;
  sin.sin_port = 0;

  prefixToNetmask(*((u_int32_t *)nic->n_net), netmask);
  inet_aton(netmask, &(sin.sin_addr));

  memcpy(&ifr.ifr_netmask, &sin, sizeof(struct sockaddr));

  // Set the netmask
  if (0 > ioctl(sock, SIOCSIFNETMASK, &ifr))
    {
      iscsi_printf("1 Could not set the netmask!\n");
      close(sock);
      return FALSE;
    }

  // Get the current flags
  memset(&ifr, 0, sizeof(struct ifreq));
  strcpy(ifr.ifr_name, nic->ifname);
  if (0 > ioctl(sock, SIOCGIFFLAGS, &ifr))
    {
      iscsi_printf("Could not get flags\n");
      close(sock);
      return FALSE;
    }

  // Set the UP flag
  ifr.ifr_flags |= IFF_UP;

  // Set the new flags, which will bring up the interface
  if (0 > ioctl(sock, SIOCSIFFLAGS, &ifr)) 
    {
      iscsi_printf("Could not start the network device\n");
      close(sock);
      return FALSE;
    }
  
  //
  // Set gateway
  //
  ipToAddress(&(((u_int32_t *)nic->n_gate)[3]), g_address);

  memset ( &route, 0, sizeof ( route ) );

  sindst.sin_family = AF_INET;
  singw.sin_family = AF_INET;

  sindst.sin_addr.s_addr = INADDR_ANY;
  inet_aton(g_address, &(singw.sin_addr));

  route.rt_dst = *(struct sockaddr *)&sindst; 
  route.rt_gateway = *(struct sockaddr *)&singw;
  route.rt_flags = RTF_GATEWAY | RTF_UP;

  // Get socket 
  if ( sockfd < 0 ) 
    {
      sockfd = socket ( PF_INET, SOCK_STREAM, 0 );
      if ( sockfd < 0 ) 
      {
        iscsi_printf ( "Could not create socket fd\n" );
        close(sock);
        return -1;
      }
    }
   
  if( ioctl(sockfd, SIOCADDRT, &route ) < 0 )
    {
      iscsi_printf("Adding Default Gateway failed\n");
      close(sock);
      return FALSE;
    }

  close(sock);
  
  return TRUE;
}

BOOL determineInitiatorOffsets(offset_t* offset)
{
  offset->i_iqn = offset->table + initiator->initiator_name_off;

  //is NIC 1 enabled?
  if (0x1 & *((u_int8_t*)(offset->table + nic1->hdr.flags)))
    {
      offset->nic_1.enabled = TRUE;
      
      //is NIC 1 primary?
      if (0x2 & *((u_int8_t*)(offset->table + nic1->hdr.flags)))
	{
	  offset->nic_1.primary = TRUE;
	}
      else
	{
	  offset->nic_1.primary = FALSE;
	}

      offset->nic_1.n_ip    = offset->table + (unsigned long)nic1->ip_addr;
      offset->nic_1.n_mac   = offset->table + (unsigned long)nic1->mac;
      offset->nic_1.n_net   = offset->table + nic1->subnet_mask_prefix;
      offset->nic_1.n_gate  = offset->table + (unsigned long)nic1->gateway;
    }
  else
    {
      offset->nic_1.enabled = FALSE;
    }

  //is NIC 0 enabled?
  if (0x1 & *((u_int8_t*)(offset->table + nic0->hdr.flags)))
    {
      offset->nic_0.enabled = TRUE;

      //is NIC 0 primary?
      if (0x2 & *((u_int8_t*)(offset->table + nic0->hdr.flags)))
	{
	  offset->nic_0.primary = TRUE;
	}
      else
	{
	  offset->nic_0.primary = FALSE;
	}

      offset->nic_0.n_ip   = offset->table + (unsigned long)nic0->ip_addr;
      offset->nic_0.n_mac  = offset->table + (unsigned long)nic0->mac;
      offset->nic_0.n_net  = offset->table + nic0->subnet_mask_prefix;
      offset->nic_0.n_gate = offset->table + (unsigned long)nic0->gateway;
    }
  else
    {
      offset->nic_0.enabled = FALSE;
    }

  if (FALSE == offset->nic_0.enabled && FALSE == offset->nic_1.enabled)
    {
      iscsi_printf("OptionROM did not report either NIC as enabled, assuming eth0\n");
      return FALSE;
    }
  
  return TRUE;
}

BOOL determineTargetOffsets(offset_t* offset)
{
  if (0x02 & *((u_int8_t*)(offset->table + tgt1->hdr.flags)))
    {
      offset->t_ip   = offset->table + (unsigned long)tgt1->ip_addr;
      offset->t_port = offset->table + tgt1->port;
      offset->t_iqn  = offset->table + *((u_int16_t*)(offset->table + tgt1->tgt_name_off));
 
      if (0x01 & *((u_int8_t*)(offset->table + tgt1->chap_type)))
	{
	  offset->chap   = TRUE;
	  offset->user   = offset->table + *((u_int16_t*)(offset->table + tgt1->chap_name_off));
	  offset->secret = offset->table + *((u_int16_t*)(offset->table + tgt1->chap_secret_off));
	}
      else
	{
	  offset->chap = FALSE;
	}
    }
  else
    {
      offset->t_ip   = offset->table + (unsigned long)tgt0->ip_addr;
      offset->t_port = offset->table + tgt0->port;
      offset->t_iqn  = offset->table + *((u_int16_t*)(offset->table + tgt0->tgt_name_off));

      if (0x01 & *((u_int8_t*)(offset->table + tgt0->chap_type)))
	{
	  offset->chap   = TRUE;
	  offset->user   = offset->table + *((u_int16_t*)(offset->table + tgt0->chap_name_off));
	  offset->secret = offset->table + *((u_int16_t*)(offset->table + tgt0->chap_secret_off));
	}
      else
	{
	  offset->chap = FALSE;
	}
      
      if (!(0x02 & *((u_int8_t*)(offset->table + tgt0->hdr.flags))))
	{
	  iscsi_printf("OptionROM did not report either target as selected, assuming eth0\n");
	  return FALSE;
	}
    }

  return TRUE;
}

BOOL ipToAddress(const u_int32_t* ip, char* address)
{
  u_int32_t a = (((u_int8_t*)ip)[0]);
  u_int32_t b = (((u_int8_t*)ip)[1]);
  u_int32_t c = (((u_int8_t*)ip)[2]);
  u_int32_t d = (((u_int8_t*)ip)[3]);

  sprintf(address, "%u.%u.%u.%u\n", a, b, c, d);
  return TRUE;
}

BOOL prefixToNetmask(u_int32_t prefix, char* netmask)
{
  u_int32_t mask = 0;
  int       index;
  
  for (index = prefix ; index > 0 ; index--)
    {
      mask = mask << 1;
      mask |= 0x01;
    }

  mask = mask << (32 - prefix);

  sprintf(netmask, "%u.%u.%u.%u\n", 
	  ((0xFF000000 & mask) >> 24),
	  ((0x00FF0000 & mask) >> 16),
	  ((0x0000FF00 & mask) >> 8),
	  ((0x000000FF & mask)));

  return TRUE;
}

void printValues(const offset_t* offset)
{
  char netmask[MAX_NETMASK_LENGTH];
  struct ether_addr eth;

  printf("**********************************************************\n");
  
  if (TRUE == offset->nic_0.enabled)
    {
      printf("NIC 0\n");
      printf(" ifname       = %s\n", offset->nic_0.ifname);
      if (TRUE == offset->nic_0.primary)
	{
	  printf(" primary      = primary\n");
	}
      else
	{
	  printf(" primary      = NOT primary\n");
	}
      printf(" IP           = %u.%u.%u.%u\n", 
	     ((u_int8_t*)(offset->nic_0.n_ip))[12],
	     ((u_int8_t*)(offset->nic_0.n_ip))[13],
	     ((u_int8_t*)(offset->nic_0.n_ip))[14],
	     ((u_int8_t*)(offset->nic_0.n_ip))[15]);

      prefixToNetmask(*((u_int32_t *)offset->nic_0.n_net), netmask);
      printf(" netmask      = %s\n", netmask);
      printf(" gateway      = %u.%u.%u.%u\n", 
	     ((u_int8_t*)(offset->nic_0.n_gate))[12],
	     ((u_int8_t*)(offset->nic_0.n_gate))[13],
	     ((u_int8_t*)(offset->nic_0.n_gate))[14],
	     ((u_int8_t*)(offset->nic_0.n_gate))[15]);

      memset(&eth, 0, sizeof(struct ether_addr));
      memcpy(eth.ether_addr_octet, (char*)(offset->nic_0.n_mac), MAC_LENGTH);
      printf(" MAC          = %s\n", ether_ntoa(&eth));
      printf("\n");
    }

  if (TRUE == offset->nic_0.enabled)
    {
      printf("NIC 1\n");
      printf(" ifname       = %s\n", offset->nic_1.ifname);
      if (TRUE == offset->nic_1.primary)
	{
	  printf(" primary      = primary\n");
	}
      else
	{
	  printf(" primary      = NOT primary\n");
	}
      printf(" IP           = %u.%u.%u.%u\n", 
	     ((u_int8_t*)(offset->nic_1.n_ip))[12],
	     ((u_int8_t*)(offset->nic_1.n_ip))[13],
	     ((u_int8_t*)(offset->nic_1.n_ip))[14],
	     ((u_int8_t*)(offset->nic_1.n_ip))[15]);
      prefixToNetmask(*((u_int32_t *)offset->nic_1.n_net), netmask);
      printf(" netmask      = %s\n", netmask);
      printf(" gateway      = %u.%u.%u.%u\n", 
	     ((u_int8_t*)(offset->nic_1.n_gate))[12],
	     ((u_int8_t*)(offset->nic_1.n_gate))[13],
	     ((u_int8_t*)(offset->nic_1.n_gate))[14],
	     ((u_int8_t*)(offset->nic_1.n_gate))[15]);
      
      memset(&eth, 0, sizeof(struct ether_addr));
      memcpy(eth.ether_addr_octet, (char*)(offset->nic_1.n_mac), MAC_LENGTH);
      printf(" MAC          = %s\n", ether_ntoa(&eth));
      printf("\n");
    }  


  printf(" i_iqn        = %s\n", OFFSET_TO_STRING(offset->i_iqn));
  printf("\n");

  printf(" t_ip:t_port  = %u.%u.%u.%u:%u\n", 
	 ((u_int8_t*)(offset->t_ip))[12],
	 ((u_int8_t*)(offset->t_ip))[13],
	 ((u_int8_t*)(offset->t_ip))[14],
	 ((u_int8_t*)(offset->t_ip))[15],
	 *((u_int16_t*)(offset->t_port)));
  printf(" t_iqn        = %s\n", OFFSET_TO_STRING(offset->t_iqn));

  if (TRUE == offset->chap)
    {
      printf("\n");
      printf(" user         = %s\n", OFFSET_TO_STRING(offset->user));
      printf(" secret       = %s\n", OFFSET_TO_STRING(offset->secret));
    }
  printf("**********************************************************\n");
}
